/*******************************************************************************
 * Copyright (c) 2000, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Eric W Li (IBM Corp) - bug 157676, perist trust decision
 *     Tie Li (IBM Corp) - timestamping of signed plugins
 *     Brian Lillie (IBM Corp) - Certificate Verification Logic and Timestamping
 *******************************************************************************/
package org.eclipse.update.internal.verifier;

import java.io.File;
import java.io.IOException;
import java.security.cert.Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.jar.JarFile;
import java.util.zip.ZipException;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.osgi.signedcontent.InvalidContentException;
import org.eclipse.osgi.signedcontent.SignedContent;
import org.eclipse.osgi.signedcontent.SignedContentEntry;
import org.eclipse.osgi.signedcontent.SignedContentFactory;
import org.eclipse.osgi.signedcontent.SignerInfo;
import org.eclipse.osgi.util.NLS;
import org.eclipse.update.core.ContentReference;
import org.eclipse.update.core.IFeature;
import org.eclipse.update.core.IVerificationResult;
import org.eclipse.update.core.IVerificationResult2;
import org.eclipse.update.core.IVerifier;
import org.eclipse.update.core.InstallMonitor;
import org.eclipse.update.core.JarContentReference;
import org.eclipse.update.core.Utilities;
import org.eclipse.update.core.Verifier;
import org.eclipse.update.internal.core.Messages;
import org.eclipse.update.internal.core.UpdateCore;

/**
 * The JarVerifier will check the integrity of the JAR.
 * If the Jar is signed and the integrity is validated,
 * it will check if one of the certificate of each file
 * is in one of the keystore.
 *
 */

public class CertVerifier extends Verifier {

	private CertVerificationResult result;
	private IProgressMonitor monitor;
	private File jarFile;
	private SignedContentFactory factory;


	/*
	 * Default Constructor
	 */
	public CertVerifier(SignedContentFactory factory) {
		this.factory = factory;
		initialize();
	}


	/*
	 * 
	 */
	private void initialize() {
		result = null;
	}

	/*
	 * init
	 */
	private void init(IFeature feature, ContentReference contentRef) throws CoreException {
		jarFile = null;
		if (contentRef instanceof JarContentReference) {
			JarContentReference jarReference = (JarContentReference) contentRef;
			try {
				jarFile = jarReference.asFile();
				if (UpdateCore.DEBUG && UpdateCore.DEBUG_SHOW_INSTALL)
					UpdateCore.debug("Attempting to read JAR file:"+jarFile); //$NON-NLS-1$
			
				// # of entries
				if (!jarFile.exists()) throw new IOException();
				JarFile jar = new JarFile(jarFile);
				if (jar !=null){
					try {
						jar.close();
					} catch (IOException ex) {
						// unchecked
					}
				}
			} catch (ZipException e){
				throw Utilities.newCoreException(NLS.bind(Messages.JarVerifier_InvalidJar, (new String[] { jarReference.toString() })), e);				
			} catch (IOException e) {
				throw Utilities.newCoreException(NLS.bind(Messages.JarVerifier_UnableToAccessJar, (new String[] { jarReference.toString() })), e);
			}
		}

		result = new CertVerificationResult();
		result.setVerificationCode(IVerificationResult.UNKNOWN_ERROR);
		result.setResultException(null);
		result.setFeature(feature);
		result.setContentReference(contentRef);
	}

	/*
	 * @param newMonitor org.eclipse.core.runtime.IProgressMonitor
	 */
	private void setMonitor(IProgressMonitor newMonitor) {
		monitor = newMonitor;
	}

	/*
	 * @see IVerifier#verify(IFeature,ContentReference,boolean, InstallMonitor)
	 */
	public IVerificationResult verify(
		IFeature feature,
		ContentReference reference,
		boolean isFeatureVerification,
		InstallMonitor monitor)
		throws CoreException {

		if (reference == null)
			return result;

		// if parent knows how to verify, ask the parent first
		if (getParent() != null) {
			IVerificationResult vr =
				getParent().verify(feature, reference, isFeatureVerification, monitor);
			if (vr.getVerificationCode() != IVerificationResult.TYPE_ENTRY_UNRECOGNIZED)
				return vr;
		}

		// the parent couldn't verify
		setMonitor(monitor);
		init(feature, reference);
		result.isFeatureVerification(isFeatureVerification);

		if (jarFile!=null) {
			result = verify(jarFile.getAbsolutePath(), reference.getIdentifier());
		} else {
			result.setVerificationCode(IVerificationResult.TYPE_ENTRY_UNRECOGNIZED);
			result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_SIGNER_TRUST_BIT, false);
		}

		return result;
	}

	/*
	 * 
	 */
	private CertVerificationResult verify(String file, String identifier) {

		try {
//			CertificateVerifier verifier = factory.getSignedContent((new File(file));
			SignedContent signedContent = factory.getSignedContent(new File(file));
			
			// verify the jar file content integrity if the jar is signed
			if(signedContent.isSigned()) {
				result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_SIGNED_BIT, true);
				verifyIntegrity(signedContent, identifier);
			}
			else { 
				result.setVerificationCode(IVerificationResult.TYPE_ENTRY_NOT_SIGNED);
				result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_SIGNED_BIT, false);	
			}
			
			//if user already said yes
			result.alreadySeen(false);

		} catch (Exception e) {
			result.setVerificationCode(IVerificationResult.UNKNOWN_ERROR);
			result.setResultException(e);
		}

		if (monitor != null) {
			monitor.worked(1);
			if (monitor.isCanceled()) {
				result.setVerificationCode(IVerificationResult.VERIFICATION_CANCELLED);
			}
		}

		return result;
	}

	/*
	 * Verifies the integrity of the JAR
	 */
	private void verifyIntegrity(
			SignedContent signedContent, 
			String identifier) {
		
		verifyEntries( signedContent.getSignedEntries() );

		SignerInfo[] infos = signedContent.getSignerInfos();
		result.setSignerInfos(infos);
		
		for (int i = 0; i < infos.length; i++) {		
			boolean signerValidWhenSigned = false;
			boolean tsaSignerValidToday = false;
			boolean tsaValidWhenSigned = false;
			boolean tsaSignerTrusted = false;
			
			// get the signing time, if there is one
			Date signingTime = signedContent.getSigningTime(infos[i]);
			
			// check whether the certificate is valid today
			boolean signerValidToday = isSignerInfoValidToday( infos[i] );
			if (!signerValidToday) {

				// if the certificate is not valid today, check whether the certificate has 
				// been extended by a timestamping authority.  If there is no signing time, 
				// then we don't have a timestamp, so don't care

				if (signingTime != null) {
				
					try {
						signedContent.checkValidity(infos[i]);
						signerValidWhenSigned = true;
					} catch (CertificateExpiredException e) {
						// ignored
					} catch (CertificateNotYetValidException e) {
						// ignored
					}
				}
			}
			
			// tsa signer info
			if(signingTime != null) {
				result.setSigningTime(signingTime);					
				SignerInfo tsaSignerInfo = signedContent.getTSASignerInfo(infos[i]);				
				result.setTSASigner(tsaSignerInfo);
				result.setVerificationBit( IVerificationResult2.TYPE_ENTRY_SIGNED_TSA_BIT, true );

				tsaValidWhenSigned = isTSASignerValid(tsaSignerInfo, signingTime, false );
				tsaSignerValidToday = isTSASignerValid(tsaSignerInfo, new Date(), true );
								
				if(tsaSignerInfo.getTrustAnchor() != null) {
					tsaSignerTrusted = true;
					result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_TSA_TRUST_BIT, true);
				} else {
					safeSetVerificationCode(IVerificationResult.TYPE_ENTRY_SIGNED_UNRECOGNIZED);
					result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_TSA_TRUST_BIT, false);
				}
			}
			
			// signer info
			if(infos[i].getTrustAnchor() != null) {
				
				// trust and expiration should be independent -- if the trust anchor is set, then set
				// the appropriate bits
				result.setTrustSigner(infos[i]);
				result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_SIGNER_TRUST_BIT, true);

				if (signerValidToday) {
					safeSetVerificationCode(IVerificationResult.TYPE_ENTRY_SIGNED_RECOGNIZED);
					
					// don't reset if there is a corrupted entry code, as that is more important
					if (result.getVerificationCode() != IVerificationResult.TYPE_ENTRY_CORRUPTED) {
						result.setVerificationCode(IVerificationResult.TYPE_ENTRY_SIGNED_RECOGNIZED);
					}
				} else if (tsaSignerTrusted) {

					if (signerValidWhenSigned && tsaValidWhenSigned) {
						if (tsaSignerValidToday) {

							// need to override, but not a corrupted setting.
							if (result.getVerificationCode() != IVerificationResult.TYPE_ENTRY_CORRUPTED) {
								result.setVerificationCode(IVerificationResult.TYPE_ENTRY_SIGNED_RECOGNIZED);
							}
						} else {

							// need to override, but not a corrupted setting.
							if (result.getVerificationCode() != IVerificationResult.TYPE_ENTRY_CORRUPTED) {
								result.setVerificationCode(IVerificationResult.TSA_CERTIFICATE_EXPIRED);
							}
						}
					} 
				}
				
				// once we find a signer with a valid trust anchor, then we are done .. don't check
				// any more signers
				
				break;
								
			} else {
				// don't reset if there is a corrupted entry code, as that is more important
				if (result.getVerificationCode() != IVerificationResult.TYPE_ENTRY_CORRUPTED) {
					result.setVerificationCode(IVerificationResult.TYPE_ENTRY_SIGNED_UNRECOGNIZED);
				}
				result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_SIGNER_TRUST_BIT, false);									
			}	
		}
	}

	
	/**
	 * Only set the verification code if it hasn't already been set to something interesting
	 * @param value
	 */
	private void safeSetVerificationCode( int value )
	{
		if (result.getVerificationCode() == IVerificationResult.UNKNOWN_ERROR) {
			result.setVerificationCode( value );
		}
	}
	
	/**
	 * Only set the verification exception if it hasn't already been set to something more
	 * critical
	 * @param t
	 */
	private void safeSetException( Exception e) 
	{
		if (result.getVerificationException() == null) {
			result.setResultException(e);
		}
	}
	
	/**
	 * 
	 * @return
	 */
	
	private void verifyEntries(SignedContentEntry[] entries )
	{
		for (int i = 0; i < entries.length; i++) {
			try {
				entries[i].verify();
			} catch (InvalidContentException e) {
				result.setVerificationCode(IVerificationResult.TYPE_ENTRY_CORRUPTED );
				result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_CORRUPTED_BIT, true);
				result.setResultException(new SecurityException( e.getMessage()));
			} catch (IOException e) {
				result.setVerificationCode(IVerificationResult.TYPE_ENTRY_CORRUPTED );
				result.setVerificationBit(IVerificationResult2.TYPE_ENTRY_CORRUPTED_BIT, true);
				result.setResultException(new SecurityException( e.getMessage()));
			}
		}
	}

	private boolean isSignerInfoValidToday( SignerInfo info )
	{
		boolean isValid = false;
		try {
	        java.security.cert.Certificate acertificate[] = info.getCertificateChain();
	        for(int i = 0; i < acertificate.length; i++)
	            if(acertificate[i] instanceof X509Certificate)
                    ((X509Certificate)acertificate[i]).checkValidity();
			isValid = true;
		} catch (CertificateExpiredException e) {
			safeSetVerificationCode(IVerificationResult.JAR_CERTIFICATE_EXPIRED);
			result.setVerificationBit(IVerificationResult2.JAR_CERTIFICATE_EXPIRED_BIT, true);
			safeSetException(e);
		} catch (CertificateNotYetValidException e) {
			safeSetVerificationCode(IVerificationResult.JAR_CERTIFICATE_EXPIRED);
			result.setVerificationBit(IVerificationResult2.JAR_CERTIFICATE_NOT_YET_VALID_BIT, true);
			safeSetException(e);
		}	
		return isValid;
	}
	
	
	private boolean isTSASignerValid( SignerInfo tsaSignerInfo, Date testDate, boolean setFailures )
	{
		boolean isValid = false;
		try {
			Certificate[] certs = tsaSignerInfo.getCertificateChain();
			for (int j = 0; j < certs.length; j++) {
				if (!(certs[j] instanceof X509Certificate)) continue;						
				((X509Certificate) certs[j]).checkValidity( testDate );						
			}
			isValid = true;
		} catch (CertificateExpiredException e) {
			if (setFailures) {
				safeSetVerificationCode(IVerificationResult.TSA_CERTIFICATE_EXPIRED);
				result.setVerificationBit(IVerificationResult2.TSA_CERTIFICATE_EXPIRED_BIT, true);
				safeSetException(e);
			}
		} catch (CertificateNotYetValidException e) {
			if (setFailures) {
				safeSetVerificationCode(IVerificationResult.TSA_CERTIFICATE_EXPIRED);
				result.setVerificationBit(IVerificationResult2.TSA_CERTIFICATE_NOT_YET_VALID_BIT, true);
				safeSetException(e);
			}
		}
		
		return isValid;
	}
			
	/**
	 * @see IVerifier#setParent(IVerifier)
	 */
	public void setParent(IVerifier parentVerifier) {
		super.setParent(parentVerifier);
		initialize();
	}

}
