/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.rcp.dombrowser.browser;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.net.ssl.X509TrustManager;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.*;
import org.osgi.service.prefs.Preferences;
import org.w3c.dom.html.HTMLDocument;

import com.ibm.rcp.dombrowser.dom.html.JHTMLWindow;
import com.ibm.rcp.dombrowser.internal.*;
import com.ibm.rcp.dombrowser.internal.mozilla.*;
import com.ibm.rcp.dombrowser.preference.IllegalPreferenceException;
import com.ibm.rcp.dombrowser.preference.Preference;
import com.ibm.rcp.dombrowser.preference.PreferencesService;

public class DOMBrowser extends DOMMozillaBrowser {
	

	private IWebHistory webHistory=null;
	boolean ignoreAllMessages = false;
    private static final String _clazzName = DOMBrowser.class.getName();
    private static final Logger logger = Logger.getLogger(DOMBrowser.class.getPackage().getName());
    
    private boolean isSSLCertTrusted = false;
    private static String CLAZZ_NAME = DOMBrowser.class.getName(); 
    private final String SERVICE_SSL_CHECK = "SSL_CERTIFICATES_CHECK_FOR_SERVICE";//$NON-NLS-1$
    
    class SSLValidatingClass{
	   	 
	   	 private List<byte[]> certArray = null;
	   	 private boolean isTrusted = false;
	   	 private final String INNER_CLAZZ_NAME = SSLValidatingClass.class.getName();
	   	 
	   	 public SSLValidatingClass(List<byte[]> certArray){
	   		 this.certArray = certArray;
	   	 }
	   	 
        /**
         * Validate the IDP server certificate chain
         * @return true if the certificate is trusted by Notes
         */
	   	public boolean checkSSLCertTrust(){
	   		final String _method = "checkSSLCertTrust";
	   		logger.entering(INNER_CLAZZ_NAME, _method);
	   		
				//build X509Certificate chain array
	   		List<X509Certificate> certChain = new ArrayList<X509Certificate>();
				Iterator it = certArray.iterator();
				while(it.hasNext()){
					byte[] cert = (byte[])it.next();
					if(cert != null && cert.length > 0){
						X509Certificate x509Cert = buildX509Certificate(cert);
						if(x509Cert != null){
							certChain.add(x509Cert);
						}
					}
				}    		
	   		
				if(certChain.size() > 0){
					
					String authType = SERVICE_SSL_CHECK;
	
					X509TrustManager x509TrustManagerService = null;
					if(enableCrossCertificate) {
						Activator pmf = Activator.getDefault();
						if(pmf != null) x509TrustManagerService = pmf.getX509CertTrustValidateService();
						
						if(x509TrustManagerService != null) {
							try {
								x509TrustManagerService.checkServerTrusted(certChain.toArray(new X509Certificate[0]), authType);
								//an exception will be thrown if certs are not trusted, otherwise set "trusted" to true
								isTrusted = true;
							} catch (CertificateException e) {
								logger.logp(Level.FINEST, INNER_CLAZZ_NAME, _method, "Validating SSL X509Certificate failed", e.getLocalizedMessage() );//$NON-NLS-1$
							}
						} else {
							logger.logp(Level.FINEST, INNER_CLAZZ_NAME, _method, "SSL X509certificate validating service is not available" );//$NON-NLS-1$
						}
					}
					
				}
	   		logger.exiting(INNER_CLAZZ_NAME, _method, Boolean.valueOf(isTrusted) );
	   		return isTrusted;
	   	}       	
	   	
	   	/*
	   	 * The X509CertImpl object is JVM specific. 
	   	 * So we need to consider both IBM JVM and non-IBM JVM cases.
	   	 */
	   	private X509Certificate buildX509Certificate(byte[] cert){
	   		final String _method = "buildX509Certificate";
	   		logger.entering(INNER_CLAZZ_NAME, _method);
	   		Class cByteArray = byte[].class;
	   		Class cDerValue = null;
	   		Class cX509CertImpl = null;
	   		X509Certificate x509Cert = null;
				try {
					cDerValue = Class.forName("sun.security.util.DerValue");
					cX509CertImpl = Class.forName("sun.security.x509.X509CertImpl");
					logger.logp(Level.FINEST, INNER_CLAZZ_NAME, _method, "Non-IBM JVM is used");//$NON-NLS-1$
					Constructor consDer = cDerValue.getConstructor(cByteArray);
					Object objDer = consDer.newInstance(cert);
					if( cDerValue.isInstance(objDer)){
						Constructor consCertImpl = cX509CertImpl.getConstructor(cDerValue);
						Object x509CertImpl = consCertImpl.newInstance(objDer);
						if( cX509CertImpl.isInstance(x509CertImpl) ){
							x509Cert = (X509Certificate)x509CertImpl;
						}
					}
				} catch (Throwable t){
					logger.logp(Level.FINEST, INNER_CLAZZ_NAME, _method, "Exception", t);//$NON-NLS-1$
				}
				logger.exiting(INNER_CLAZZ_NAME, _method, x509Cert.toString());
				return x509Cert;
	   	}
   	 
    }
    	
    public DOMBrowser(Composite parent, int style) {
        super(parent,style); 
        convertEclipsePreference2MozillaPreference();
        
        // add listen to valid server certificate with Notes/XPD CA database. 
        if(enableCrossCertificate) {
        	modalContextProgress = new NullProgressMonitor();
        	this.addCertificateErrorListener(new CertificateErrorListener() {
    			public void handleCertificateError(CertificateErrorEvent event) {
    				event.doit = false;	//initial value false;
    				List<byte[]> certArray = event.rawCertInDERArray;
    				if(certArray != null && certArray.size() > 0){
    					//SPR # KKSS92W4WF && KKSS92VM9C
    					//use ModalContext to validate SSL certificate
    					doSSLValidationForServerCert(certArray);
    					if(isSSLCertTrusted){
    						event.doit = true;
    						return;
    					}
    				}
    			}
            });
        	
        }
    }	
    
    private IProgressMonitor modalContextProgress = null;
    
    private void doSSLValidationForServerCert(final List<byte[]> certArray){
		 final String _method = "doSSLValidationForServerCert";
		 logger.entering(CLAZZ_NAME, _method, Boolean.valueOf(isSSLCertTrusted));
		
		 try {
			 ModalContext.run(new IRunnableWithProgress( ) {
		
				 public void run(IProgressMonitor progress)	throws InvocationTargetException, InterruptedException {
					 SSLValidatingClass sslValidate = new SSLValidatingClass(certArray);
					 if(sslValidate.checkSSLCertTrust()){
						 isSSLCertTrusted = true;
					 }
				 }
		
			 }, true, modalContextProgress, Display.getDefault( ));
		
		 } catch (Throwable e1) { 
			 logger.logp(Level.FINEST, CLAZZ_NAME, _method, "exception from ModalContext.run()", e1);//$NON-NLS-1$
		 }
		
		 logger.exiting(CLAZZ_NAME, _method, Boolean.valueOf(isSSLCertTrusted));
    }

    
    protected MozillaEmbeddingSite createEmbeddingSite() {
    	return new DOMMozillaEmbeddingSite(this);
    }    		
       
	protected WeakReferenceManager createWeakReferenceManager() {
		return new WeakReferenceManagerex();
	}
	

    
	/**
	 * Cleanup
	 */
	protected void onDispose(Display display) {
		// Add below code to fire onbeforeunload event
		// Commenting on purpose as seturl is bound to fail during loading of the page after dispose is invoked
		// syncing as in swt Mozilla code
//		LocationListener[] oldLocationListeners = locationListeners;
//		locationListeners = new LocationListener[0];
//		ignoreAllMessages = true;
//		//execute("window.location.replace('about:blank');"); //$NON-NLS-1$
//		//For SPR#GGAR8F9L2B, we take setUrl("about:blank") instead.
//		setUrl("about:blank");
//		ignoreAllMessages = false;
//		locationListeners = oldLocationListeners;
		
	    super.onDispose(display);
	}	

	
	static public int getBrowser(int /*long*/ aDOMWindow,Composite[] browser) {
		browser[0] = null;
		if (aDOMWindow == 0)
			return XPCOM.NS_ERROR_NULL_POINTER;
		
		nsIDOMWindow window = new nsIDOMWindow(aDOMWindow);
		int /*long*/[] result = new int /*long*/[1];
		int rc = window.GetTop(result);
		if (rc != XPCOM.NS_OK) 
			return rc;
		
		if (result[0] == 0) 
			return XPCOM.NS_NOINTERFACE;
		
		nsIDOMWindow topWindow = new nsIDOMWindow(result[0]);
		
		result[0] = 0;
		rc = XPCOM.NS_GetServiceManager(result);
		if (rc != XPCOM.NS_OK) {
			topWindow.Release();
			return rc;
		}	
		
		if (result[0] == 0) {
			topWindow.Release();
			return XPCOM.NS_NOINTERFACE;
		}	
		
		nsIServiceManager serviceManager = new nsIServiceManager(result[0]);
		result[0] = 0;
		byte[] buffer = XPCOM.NS_WINDOWWATCHER_CONTRACTID.getBytes();
		byte[] aContractID = new byte[buffer.length + 1];
		System.arraycopy(buffer, 0, aContractID, 0, buffer.length);
		rc = serviceManager.GetServiceByContractID(aContractID, nsIWindowWatcher.NS_IWINDOWWATCHER_IID, result);
		serviceManager.Release();
		if (rc != XPCOM.NS_OK) {
			topWindow.Release();
			return rc;
		}	
		
		if (result[0] == 0) {
			topWindow.Release();
			return XPCOM.NS_NOINTERFACE;
		}	
		
		nsIWindowWatcher windowWatcher = new nsIWindowWatcher(result[0]);
		result[0] = 0;
		rc = windowWatcher.GetChromeForWindow(topWindow.getAddress(), result);
		topWindow.Release();
		windowWatcher.Release();	
		if (rc != XPCOM.NS_OK) 
			return rc;
		
		if (result[0] == 0) 
			return XPCOM.NS_NOINTERFACE;	
				
		nsIWebBrowserChrome webBrowserChrome = new nsIWebBrowserChrome(result[0]);
		result[0] = 0;
		rc = webBrowserChrome.QueryInterface(nsIEmbeddingSiteWindow.NS_IEMBEDDINGSITEWINDOW_IID, result);
		webBrowserChrome.Release();
		if (rc != XPCOM.NS_OK) 
			return rc;
		
		if (result[0] == 0) 
			return XPCOM.NS_NOINTERFACE;		
				
		nsIEmbeddingSiteWindow embeddingSiteWindow = new nsIEmbeddingSiteWindow(result[0]);
		result[0] = 0;
		//SPR # RRANBJRAB8
		if (!XPCOM.isMacOSX) {
			MozillaBrowser.IsGettingSiteWindow = true;
		}
		rc = embeddingSiteWindow.GetSiteWindow(result);
		//SPR # RRANBJRAB8
		if (!XPCOM.isMacOSX) {
			MozillaBrowser.IsGettingSiteWindow = false;
		}
		embeddingSiteWindow.Release();
		if (rc != XPCOM.NS_OK) 
			return rc;
		
		if (result[0] == 0) 
			return XPCOM.NS_NOINTERFACE;
		
		browser[0] = findBrowser(result[0]); 
		return rc;
	}

	public void addDocumentCompleteListener(DocumentCompleteListener listener) {
		super.addDocumentCompleteListener(listener);
	}

	public HTMLDocument getDocument() {
		return super.getDocument();
	}

	public JHTMLWindow getHTMLWindow() {
		return super.getHTMLWindow();
	}

	public void removeDocumentCompleteListener(DocumentCompleteListener listener) {
		super.removeDocumentCompleteListener(listener);
	}

	public void addCloseWindowListener(CloseWindowListener listener) {
		super.addCloseWindowListener(listener);
	}

	public void addLocationListener(LocationListener listener) {
		super.addLocationListener(listener);
	}

	public void addOpenWindowListener(OpenWindowListener listener) {
		super.addOpenWindowListener(listener);
	}

	public void addProgressListener(ProgressListener listener) {
		super.addProgressListener(listener);
	}

	public void addStatusTextListener(StatusTextListener listener) {
		super.addStatusTextListener(listener);
	}

	public void addTitleListener(TitleListener listener) {
		super.addTitleListener(listener);
	}

	public void addVisibilityWindowListener(VisibilityWindowListener listener) {
		super.addVisibilityWindowListener(listener);
	}

	public boolean back() {
		return super.back();
	}

	public boolean execute(String script) {
		return super.execute(script);
	}

	public boolean forward() {
		return super.forward();
	}

	public String getUrl() {
		return super.getUrl();
	}

	public void refresh() {
		super.refresh();
	}

	public void removeCloseWindowListener(CloseWindowListener listener) {
		super.removeCloseWindowListener(listener);
	}

	public void removeLocationListener(LocationListener listener) {
		super.removeLocationListener(listener);
	}

	public void removeOpenWindowListener(OpenWindowListener listener) {
		super.removeOpenWindowListener(listener);
	}

	public void removeProgressListener(ProgressListener listener) {
		super.removeProgressListener(listener);
	}

	public void removeStatusTextListener(StatusTextListener listener) {
		super.removeStatusTextListener(listener);
	}

	public void removeTitleListener(TitleListener listener) {
		super.removeTitleListener(listener);
	}

	public void removeVisibilityWindowListener(VisibilityWindowListener listener) {
		super.removeVisibilityWindowListener(listener);
	}

	public boolean setText(String html) {
		return super.setText(html);
	}

	public boolean setUrl(String url) {
		return super.setUrl(url);
	}

	public boolean setUrl(String url, String postData, String[] headers) {
		return super.setUrl(url, postData, headers);
	}
	
	
	public void stop() {
		super.stop();
	}

	public SecuritySettings getSecuritySettings() {
		return super.getSecuritySettings();
	}	

    public IWebHistory getWebHistory(){
    	if (webHistory==null){
    		webHistory=new MozillaWebHistory(this);
    	}
    	return webHistory;
    }

	public void addContextMenuListener(ContextMenuListener contextMenuListener) {
		super.addContextMenuListener(contextMenuListener);
	}

	public void removeContextMenuListener() {
		super.removeContextMenuListener();
	}

	public void addFileDownloadListener(FileDownloadListener aFileDownloadListener) {
		super.addFileDownloadListener(aFileDownloadListener);
	}
	
	public void removeFileDownloadListener() {
		super.removeFileDownloadListener();
	}

	public void removeAuthenticationListener(AuthenticationListener listener) {
		super.removeAuthenticationListener(listener);
	}
	public void addAuthenticationListener(AuthenticationListener listener) {
		super.addAuthenticationListener(listener);
	}
		
	public void removeCertificateErrorListener(CertificateErrorListener listener) {
		super.removeCertificateErrorListener(listener);
	}
	public void addCertificateErrorListener(CertificateErrorListener listener) {
		super.addCertificateErrorListener(listener);
	}
	
	public void removeAlertListener(AlertListener listener) {
		super.removeAlertListener(listener);
	}
	public void addAlertListener(AlertListener listener) {
		super.addAlertListener(listener);
	}
	
	public void addPromptListener(PromptListener listener) {
		super.addPromptListener(listener);
	}
	public void removePromptListener(PromptListener listener) {
		super.removePromptListener(listener);
	}
	
	public void addConfirmListener(ConfirmListener listener) {
		super.addConfirmListener(listener);
	}
	
	public void removeConfirmListener(ConfirmListener listener) {
		super.removeConfirmListener(listener);
	}

	public void updateSSOStatus(String statusString) {
	    final String _methodName = "DOMBrowser::updateSSOStatus()";
		try{
			StatusTextEvent event = new StatusTextEvent(DOMBrowser.this);
			event.display = Display.getDefault();
			event.widget = DOMBrowser.this;
			event.text = statusString;
			for (int i = 0; i < this.statusTextListeners.length; i++){
				this.statusTextListeners[i].changed(event);
			}
		}catch(Exception e){
			logger.logp(Level.WARNING, _clazzName, _methodName, "Failed to do SSO.", e);  //$NON-NLS-1$
		}
	}


	private boolean SSOCancel=false;
	/*
	 * while user clicks cancel, return is true; Otherwise return false;
	 */
	public boolean isSSOCancel() {
		return SSOCancel;
	}
	public void setSSOCancel(boolean cancelable) {
		SSOCancel = cancelable;
	}	
	
	public boolean doesSelectionExist() {
		return super.doesSelectionExist();
	}
	public String getSelectedText() {
		return super.getSelectedText();
	}
	
	private static final String DOMBROWSER_PLUGIN_ID					= "com.ibm.rcp.dombrowser";								//$NON-NLS-1$
	private static final String ENABLE_ECLIPSE_TO_MOZILLA_PREFERENCE 	= "enableEclipseToMozillaPreference";					//$NON-NLS-1$
	private static final String MOZILLA_PREFERENCE_NODE 				= "com.ibm.rcp.dombrowser.mozillaPreference";			//$NON-NLS-1$
	private static final String PREFERENCE_SET_INT_TYPE_PREFIX			= "set,I,";												//$NON-NLS-1$
	private static final String PREFERENCE_SET_BOOLEAN_TYPE_PREFIX		= "set,B,";												//$NON-NLS-1$
	private static final String PREFERENCE_SET_STRING_TYPE_PREFIX		= "set,S,";												//$NON-NLS-1$
	private static final String PREFERENCE_DELETE						= "delete";												//$NON-NLS-1$
	private static final String PREFERENCE_RESET						= "reset";												//$NON-NLS-1$
	private static final String PREFERENCE_LOCK							= "lock";												//$NON-NLS-1$
	private static final String PREFERENCE_UNLOCK						= "unlock";												//$NON-NLS-1$
	
	private static boolean hasConvertedFromEclipseToMozillaPreference = false;
	
	private boolean isPreferenceConvertEnabled(){
		return Platform.getPreferencesService().getBoolean(DOMBROWSER_PLUGIN_ID, ENABLE_ECLIPSE_TO_MOZILLA_PREFERENCE, false, null);
	}
	
	// convert the eclipse preference from DefaultScope, ConfigurationScope and InstanceScope to mozilla preference
	private void convertEclipsePreference2MozillaPreference(){
		if(hasConvertedFromEclipseToMozillaPreference) {
			return;
		}else{
			hasConvertedFromEclipseToMozillaPreference = true;
		}
		if(isPreferenceConvertEnabled()){
			convert2MozillaPreference(Platform.getPreferencesService().getRootNode());
		}
	}
	
	/*
	 * Instance scope preference will overwrite the one in Default Scope
	 */
	private Map<String, String> getAllPreferenceKeysToBeConverted(IEclipsePreferences root){
		String _methodName = "getAllPreferenceKeysToBeConverted";  //$NON-NLS-1$
		HashMap<String, String> retVMap = new HashMap<String, String>();
		String[] scopes = new String[]{DefaultScope.SCOPE, InstanceScope.SCOPE}; 
		Preferences prefNode = null;
		int scopeLength = scopes.length;
		for(int i=0; i< scopeLength; i++){
			String contextScope = scopes[i];
			try {
				prefNode = root.node(contextScope).node(MOZILLA_PREFERENCE_NODE);
				if(prefNode == null){
					continue;
				}
				String[] preferenceKeys = prefNode.keys(); // which can't be null
				if(preferenceKeys.length > 0){
					int keyCount = preferenceKeys.length;
					String key = null;
					for(int j=0; j< keyCount; j++){
						key = preferenceKeys[j];
						if(key == null){
							continue;
						}
						key = key.trim();
						if(key.length()==0){
							continue;
						}
						retVMap.put(key, prefNode.get(key, null));
					}
				}
			} catch (Exception e) {
				logger.logp(Level.WARNING, _clazzName, _methodName, NLS.bind(DOMBrowserNLS.Error_Fail_Convert_EclipsePreference_To_MozillaPreference, MOZILLA_PREFERENCE_NODE), e);
			}
		}
		return retVMap;
	}
	
	private void convert2MozillaPreference(IEclipsePreferences root){
		String _methodName = "convert2MozillaPreference";  //$NON-NLS-1$
		PreferencesService prefService = null;
		try {
			prefService = PreferencesService.getService();
		} catch(Exception e) {
			logger.logp(Level.WARNING, _clazzName, _methodName, NLS.bind(DOMBrowserNLS.Error_Fail_Convert_EclipsePreference_To_MozillaPreference, MOZILLA_PREFERENCE_NODE), e);
		}
		if(prefService == null){
			return;
		}
		Map<String, String> allKeysNeedtoConvert = getAllPreferenceKeysToBeConverted(root);
		if(allKeysNeedtoConvert == null || allKeysNeedtoConvert.size() == 0){
			return;
		}
		Preference pref = null;
		int prefixLength = PREFERENCE_SET_INT_TYPE_PREFIX.length();
		Iterator<Entry<String, String>> itor = allKeysNeedtoConvert.entrySet().iterator();
		while(itor.hasNext()){
			Entry<String, String> entry = itor.next();
			String key = entry.getKey();
			try {
				String strValue = entry.getValue();
				if(strValue == null){
					continue;
				}
				if(strValue.startsWith(PREFERENCE_SET_INT_TYPE_PREFIX)){
					// set integer preference into mozilla
					int intPrefValue = Integer.parseInt(strValue.substring(prefixLength));
					pref = prefService.createUserPref(key, PreferencesService.PREF_INT);
					pref.setPrefIntValue(intPrefValue);
					prefService.setUserPref(pref);
				}else if(strValue.startsWith(PREFERENCE_SET_BOOLEAN_TYPE_PREFIX)){
					// set boolean preference into mozilla
					boolean bPrefValue = Boolean.parseBoolean(strValue.substring(prefixLength));
					pref = prefService.createUserPref(key, PreferencesService.PREF_BOOL);
					pref.setPrefBooleanValue(bPrefValue);
					prefService.setUserPref(pref);
				}else if(strValue.startsWith(PREFERENCE_SET_STRING_TYPE_PREFIX)){
					// set string preference into mozilla
					pref = prefService.createUserPref(key, PreferencesService.PREF_STRING);
					pref.setPrefStrValue(strValue.substring(prefixLength));
					prefService.setUserPref(pref);
				}else if(strValue.equals(PREFERENCE_DELETE)){
					// delete the preference from mozilla
					pref = prefService.getUserPref(key);
					prefService.deletePref(pref);
				}else if(strValue.equals(PREFERENCE_RESET)){
					// clear the preference from mozilla
					pref = prefService.getUserPref(key);
					prefService.clearUserPref(pref);
				}else if(strValue.equals(PREFERENCE_LOCK)){
					// lock the preference in mozilla
					pref = prefService.getUserPref(key);
					prefService.lockPref(pref);
				}else if(strValue.equals(PREFERENCE_UNLOCK)){
					// unlock the preference in mozilla, not work for user preference
					pref = prefService.getUserPref(key);
					prefService.unlockPref(pref);
				}
			}
			catch (NumberFormatException e){
				logger.logp(Level.WARNING, _clazzName, _methodName, NLS.bind(DOMBrowserNLS.Error_Invalid_Preference_Type, key), e);
			}
			catch (IllegalPreferenceException e) {
				if(e.getCauseID() == IllegalPreferenceException.INVALID_PREFERENCE_TYPE){
					logger.logp(Level.WARNING, _clazzName, _methodName, NLS.bind(DOMBrowserNLS.Error_Invalid_Preference_Type, key), e);
				}
			}catch (Exception e) {
				logger.logp(Level.WARNING, _clazzName, _methodName, NLS.bind(DOMBrowserNLS.Error_Fail_Convert_EclipsePreference_To_MozillaPreference, key), e);  
			} 
		}
		allKeysNeedtoConvert.clear(); //clear the hash map.
		allKeysNeedtoConvert = null;
	}
}	
