# /unix/nui/installer/tools/lib/V9/InstBEISMP.pl
#
# Used for Domino 11 fixpacks
# Used for Domino 12 fixpacks
#
# ***************************************************************** 
#                                                                   
# HCL Confidential                                                  
#                                                                   
# OCO Source Materials                                              
#                                                                   
# Copyright HCL 2022
#                                                                   
# The source code for this program is not published or otherwise    
# divested of its trade secrets, irrespective of what has been      
# deposited with the U.S. Copyright Office.                         
#                                                                   
# ***************************************************************** 
#
## Copyright (c) 1998, Lotus Development Corporation.
## All Rights Reserved.
#
#use warnings; # Because of $Sys::xxx 
use strict;
 
use v5.10;  # For 'say'

# Turn off buffering of STDOUT
$|=1;
 
$SIG{'INT'} = $SIG{'QUIT'} = $SIG{'TERM'} = 'IGNORE';
 
# Load the CdPath package, used for Require-ing other perl files,
# and to translate misc filenames to a form accesible on the cd.
# @INC is already set appropriately here.
# So that this multiple eval junk here does not need to be repeated in
# every perl file that requires another, member functions are to be
# explicity called via CdPath::
#
eval 'require "CdPath.pl";';
eval 'require "CDPATH.PL";';
eval 'require "CDPATH.PL;1";';
eval 'require "cdpath.pl";';
CdPath::CdPathSetRoot($ENV{NUI_DISTROOT});

 
CdPath::Require("CfgData.pm");		import CfgData;
CdPath::Require("CDFFile.pm");		import CDFFile;
CdPath::Require("PerlUtil.pl");		import PerlUtil;
CdPath::Require("InstBE.nls");		import InstBE_nls;
CdPath::Require("NIC.pm");			import NIC;
CdPath::Require("ShScript.pm");		import ShScript;
CdPath::Require("SysCmd.pl");

my $dbg = $ENV{'NUI_DBG_BE'}; # Setting to 1 will produce debug, 2 is verbose debug

say "InstBE.pl: Starting" if $dbg;

#
# Setup & check system commands before anything else
#
my @reqcmds = qw(
	mv rm mkdir env date touch tar chown cp hostname clear
	chgrp cat chmod sh ln pwd ls dirname basename
);

#
# Determine how to uncompress
#
my $uncompressCmd;

if ( $ENV{'NUI_ARCH'} =~ /linux/ ) {
	push @reqcmds, qw(gunzip);
	$uncompressCmd = "gunzip";
} else {
	push @reqcmds,  qw(uncompress);
	$uncompressCmd = "uncompress";
}

my @cmdsnotfound;
my $errmsg;
if ( ! SysCmd::InitCmds(\@reqcmds, \@cmdsnotfound, \$errmsg, $ENV{'NUI_ARCH'})) {
	Out("InstBE.pl: Fatal Error:" . $errmsg);
	Exit(1);
}

#
# Constants
#
my $cdffile = CdPath::CdPath("sets/data/comp.cfg");
my $fgmapfile = CdPath::CdPath("sets/data/fg.dat");
my $fgsetdir = "sets/fgroups/";
my $fgsetext = ".taz";

#
# Variables
#
my $scrfile;
my $scrcfg;
my $scrdat;
my $cdf;
my $fgmapcfg;
my $fgmap;

#
# Parse args
#
$scrfile = $ARGV[0];

say "InstBE.pl: Passed " . $scrfile if $dbg;

#
# Load the filegroup map data
#
$fgmapcfg = new CfgData ();
if ( ! $fgmapcfg->InitFromFile($fgmapfile)) {
	Out("InstBE.pl: Fatal Error loading filegroup map file: " . $fgmapcfg->GetErrorMessage());
	Exit(1);
}
else
{
    say "InstBE.pl: Successfully initialized from " . $fgmapfile if $dbg;
}

$fgmap = $fgmapcfg->GetData();
undef $fgmapcfg;

#
# Load the nui cfg file
#
my $nuicfgfile;
my $nuicfg;
my $nuicfgdat;

$nuicfgfile = CdPath::CdPath("sets/data/nui.cfg");
$nuicfg = new CfgData ();

if ( ! $nuicfg->InitFromFile($nuicfgfile)) {
    Out("InstBE.pl: Fatal Error loading installer configuration file: " . $nuicfg->GetErrorMessage());
    Exit(1);
}
else
{
    say "InstBE.pl: Successfully initialized from " . $nuicfgfile if $dbg;
}

$nuicfgdat = $nuicfg->GetData();
undef $nuicfg;

#
# Load script
#
my $d = `$Sys::dirname $scrfile`;
chomp($d);

if ( ! -d $d) {
	say "InstBE.pl: Unable to load script file " . $scrfile;
	Exit(1);
}
else
{
	say "InstBE.pl: Successfully loaded script file " . $scrfile if $dbg;
}

chdir($d);
$d = `$Sys::pwd`; chomp($d);
$scrfile = "${d}/" . `$Sys::basename $scrfile`;
chomp($scrfile);
$scrcfg = new CfgData ();

if ( ! $scrcfg->InitFromFile($scrfile)) {
	say "InstBE.pl: Fatal Error loading script file: " . $scrfile;
	say $scrcfg->GetErrorMessage();
	Exit(1);
}
else
{
    say "InstBE.pl: Successfully initialized from " . $scrfile if $dbg;
}

$scrdat = $scrcfg->GetData();
undef($scrcfg);

#
# Load the component def file
#
$cdf = new CfgData();

if ( ! $cdf->InitFromFile($cdffile)) {
	say "InstBE.pl: Fatal Error loading component definition file:";
	say $cdf->GetErrorMessage();
	Exit(1);
}
else
{
	say "InstBE.pl: Successfully loaded component definition file:" . $cdffile if $dbg;
}

#
# Create the NIC obj
#
my $nic = new NIC ($cdf, $scrdat);

#
# Ask NIC to create the context's
#
my @ctxs;
if ( ! $nic->InitContexts(\@ctxs, $nuicfgdat)) {
	Out($nic->GetErrMsg());
	Exit(1);
}

#
# Go thought ctx's and set isLocal
#
my $ctx;

foreach $ctx (@ctxs) {
	$ctx->{isLocal} = 1;
}


if ($nic->IsBatchMode()) {
	Out(`$Sys::date` . "\n");
	say "Batch mode script file is: " . $scrfile;
}

my $stat;

foreach $ctx (@ctxs) {

	if ($ENV{NUI_DBG_INCR}) {
		say "local ctx prior to Do-ing>";
		DumpData($ctx);
		say "<";
	}
	

	#
	# We no longer support installing remotely.  Did we do assessments, get by-in
	# from above, document for customers, and all that?  Nope.  Even better.
	#
	# We performed a remote install using the command 'rsh' - it's a legacy remote 
	# shell tool that was replaced by 'ssh' a long time ago.  It's nearly impossible
	# to install this on a Linux box these days as they're sure you don't really want it.
	#
	# It has been like this for years and we've never had a customer complain.  Now it's gone.
	#
	my $msg = "";

	say "Validating...\n";
	say "InstBE.pl: Calling ValidateContext()" if $dbg;

	if ( ! $nic->ValidateContext($ctx, \$msg)) {
		Out($msg);
		say "InstBE.pl: ValidateContext() failed.  The installation cannot proceed.\n\n";

		# Exit with error
		Exit(1);
	}
	else
	{
	    say "InstBE.pl: ValidateContext() successful" if $dbg;
	}

	Out($msg);

	#
	# setup component destination data
	#
	my @destPathList;
	my %destPathCompHash;

	say "InstBE.pl: Calling GetComponentDestinationData()" if $dbg;
	$nic->GetComponentDestinationData($ctx, \@destPathList, \%destPathCompHash);

	if ($ENV{'NUI_NOINSTALL'}) {
		Out("NUI_NOINSTALL: done.\n");

		# Exit without error
		Exit(0);
	}

	say "Installing Domino Server kits...";
       
	if ($ctx->{nuicfg}->{kitType} eq 'fix'){
		my %base_checksums;
		my %target_checksums;
		my $filename;
		my %checked;

		say "InstBE.pl: Our kit type is 'fix'" if $dbg;
		
		foreach $filename (keys %{$ctx->{nuicfg}->{$ctx->{base_version}}}){
		    $base_checksums{$filename} = 
			$ctx->{nuicfg}->{$ctx->{base_version}}->{$filename}->{checksum};
		}

		foreach $filename (keys %{$ctx->{nuicfg}->{$ctx->{target_version}}}){
		    $target_checksums{$filename} = 
			$ctx->{nuicfg}->{$ctx->{target_version}}->{$filename}->{checksum};
		}

		my @fixes = keys %base_checksums;

		my $fname;
		foreach $fname (@fixes){

			if (!defined $target_checksums{$fname}){
				next;
			}
			else{
				if (($base_checksums{$fname} > 0) && ($target_checksums{$fname} > 0) &&
				    ($base_checksums{$fname} != $target_checksums{$fname})){ 
					say "Replacing " . $fname;
					$checked{$fname} = 1;
				}
				elsif (($base_checksums{$fname} == 0)  && ($target_checksums{$fname} > 0)){
				    say "Replacing " . $fname;
				    $checked{$fname} = 1;
				}
				elsif (($target_checksums{$fname} == 0) && ($base_checksums{$fname} > 0)){
				    say "Replacing " . $fname;
				    $checked{$fname} = 1;
				}
			}
		}

		@fixes = keys %target_checksums; 
		foreach $fname (@fixes){

			say "InstBE.pl: fname = " . $fname if $dbg;

			next unless (!$checked{$fname});

			if (!defined $base_checksums{$fname}){
				# Skip 'special' files we handle seperately
				next if ($fname eq "<NOTESPROGDIR>osgi_rcp_plugins.taz");
				next if ($fname eq "<NOTESPROGDIR>osgi_plugins.taz");
				next if ($fname eq "<NOTESPROGDIR>jvm_patch.diff");
				next if ($fname eq "<NOTESPROGDIR>no_fileosgi_rcp_plugins.taz");
				next if ($fname eq "<NOTESPROGDIR>no_fileosgi_plugins.taz");
				next if ($fname eq "<NOTESPROGDIR>no_filejvm_patch.diff");

				# When reverting, the 'special' files have a 'no_file' in their name like:
				#
				# Replacing <NOTESPROGDIR>no_filetika_pdf_sort.xml
				# Replacing <NOTESPROGDIR>no_filelibcrypto-domino.so.1.1
				# Replacing <NOTESPROGDIR>no_fileosgi_plugins.taz
				# Replacing <NOTESPROGDIR>no_filelibssl-domino.so.1.1
				# Replacing <NOTESPROGDIR>no_filejvm_patch.diff
				# Replacing <NOTESPROGDIR>no_fileNotes.jar
				# Replacing <NOTESPROGDIR>no_fileosgi_rcp_plugins.taz
				# Replacing <NOTESPROGDIR>no_filewebsvc.jar
				#
				# So strip that out when displaying to end-user.
				my $dispFname = $fname;
				$dispFname =~ s/no_file//; 

				# We used to say 'Adding' here for non-checksum files.  But it's too early to know if
				# if we're applying the fixpack or reverting it.  So 'Adding' or 'Deleting' is misleading.
				# Switched to 'Replacing' and what we use right above where we list the other
				# files that make up the fixpack.
				say "Replacing " . $dispFname;
				next;
			}
			else{
				if (($target_checksums{$fname} > 0) && ($base_checksums{$fname} > 0) &&
				    ($base_checksums{$fname} != $target_checksums{$fname})){ 
					say "Replacing " . $fname;
				}

				if (($base_checksums{$fname} == 0)  && ($target_checksums{$fname} >  0)){
					# Out($txt{7.3}.$fname.$txt{4.1});
					next;
				}

				if (($target_checksums{$fname} == 0) && ($base_checksums{$fname} > 0)){
					# Out($txt{7.4}.$fname.$txt{4.1});
					next;
				}
			}
		}
	}

	#
	# Pre-install processing
	#
	say "InstBE.pl: Calling PreInstallProcessing()" if $dbg;

	if ( ! $nic->PreInstallProcessing($ctx, \$msg)) {
		say "Unexpected PreInstallProcessing failure: ";
		say $msg;
		say "The installation failed.";

		# Exit with error
		Exit(1);
	} else {
		Out($msg);
		say "InstBE.pl: PreInstallProcessing() successful" if $dbg;
	}

	#added for hotfixes
	if (($ctx->{nuicfg}->{kitType} eq 'fix') && ($ctx->{fix_type} eq 'revert')){

	    say "InstBE.pl: Calling PostInstallProcessing()" if $dbg;

	    if ( ! $nic->PostInstallProcessing($ctx, \$msg)) {
			Out($msg);

			# Exit with error
			Exit(1);
	    }
	    else
	    {
			say "InstBE.pl: PostInstallProcessing() successful" if $dbg;
	    }

	    Success();
	}
	
	#
	# local install stuff
	#
	my $destPath;
	my $comp;
	my $fglist;
	my $fg;
	my $fgset;
	my $cmd;
	my $out;
	my $firstTime = 1;

	foreach $destPath (@destPathList) {

		say "Processing files to be placed in: " . $destPath;

		if ( ! -d $destPath ) {
			if (system("$Sys::mkdir -p $destPath")) {
				say "Unable to create directory " . $destPath . $!;

				# Exit with error
				Exit(1);
			}
		}

		foreach $comp (@{$destPathCompHash{$destPath}}) {

			$fglist = $cdf->{data}->{_KeyList_};

			if ($dbg && $firstTime) {
				$firstTime = 0;
				print "  filegroups:";
				DumpData($fglist, 2);
			}

			$fg = $comp;

			if ($fgmap->{$fg} eq "no_files") {
				if ($dbg) {
					say "  NOT doing file group $fg: no files";
				}
				next;
			}

			say "  Doing file group " . $fg if $dbg;
		
			$fgset = CdPath::CdPathShellString($fgsetdir . $fgmap->{$fg} . $fgsetext);

			say "    fgset: " . $fgset if $dbg;

			if (! $ENV{'NUI_NOUNTAR'}) {
				my %fgroup;
				my %fuid;
				my %fmode;
				my @sb = stat($destPath);
				$fgroup{$destPath} = $sb[5];
				$fuid{$destPath} = $sb[4];
				$fmode{$destPath} = $sb[2] & 07777;

				say "    Extracting files from " . $fgset;

				$cmd = 	"cd $destPath; " .
					"$Sys::cat $fgset | " .
					"$uncompressCmd -c | $Sys::tar xvf -";

				say "    extract cmd: $cmd" if $dbg;
				say "    extracting " . `date` if $dbg;

				$out = `$Sys::sh -c "$cmd" 2>&1`;
				$stat = ($? >> 8);

				say "    stat: " . $stat if $dbg;
				# This is very verbose as it lists each file, and for osgi and other directories it is huge:
				say "    out: " . $out if ($dbg > 1);

				if ($stat) {
					say "Fatal Error extracting filegroup: " . $cmd;
					say $out;

					# Exit with error
					Exit(1);
				}

				chown($fuid{$destPath}, $fgroup{$destPath}, $destPath) or 
 					die "can't change owner back on $destPath:$!";
				system("$Sys::chown -R $fuid{$destPath}:$fgroup{$destPath}  $destPath");
				    
				say "    done " . `date` if $dbg;
			}
		}
	
		say "\n" if $dbg;
	}

	#
	# Post-install processing
	#
	say "InstBE.pl: Calling PostInstallProcessing() 2" if $dbg;

	if (! $nic->PostInstallProcessing($ctx, \$msg)) {
		say "An unexpected error occured during post-install processing: " . $msg;
		say "The installation failed.";

		# Exit with error
		Exit(1);
	} else {
		say "InstBE.pl: PostInstallProcessing() 2 successful" if $dbg;
		Out($msg);
	}

	Success(); # Will print success and then exit
}

######################################################################
#
# End of main script
#
######################################################################

sub Success {
	say "\n";
	say "The installation completed successfully.\n";
	say "Please be sure to login as the appropriate UNIX user";
	say "before running Domino - Do not run as root\n\n";
	exit 0;
}

sub Out {
	print @_;
}


sub Exit {
	my $stat = @_;

	exit $stat;
}
