#!/usr/bin/perl
# ***************************************************************** 
#                                                                   
# HCL Confidential                                                  
#                                                                   
# OCO Source Materials                                              
#                                                                   
# Copyright HCL 2022
#                                                                   
# The source code for this program is not published or otherwise    
# divested of its trade secrets, irrespective of what has been      
# deposited with the U.S. Copyright Office.                         
#                                                                   
# ***************************************************************** 



######################################################################
#
# MAIN PROGRAM - CheckOS
#
# This Perl Script supports the following options:
#
#  checkos
#
#       exit status:    Number of missing OS patches
#
#	-f <name>:	Use <name> as patch database
#	-h:  		Show the command line options
#	-d:         Print debugging information\n";
#
# NOTE: This script, along with checkos.dat, is used by both gold
#       installs on Linux/AIX as well as fixpacks on these platforms.
#
#
######################################################################
#
#	MAIN PROGRAM
#
######################################################################

#
# Non-dependency on package main
#
package HclDominoCheckOS;

use warnings;
use strict;

#
# Enable debug
#
my $debug = 0;

if (defined $ENV{INSTALL_DEBUG})
{
	$debug = 1;
}

#
# Flush output right away
#
$|=1;

my $candidate;

my @currentPackageList;
my @requiredPackageList;
my @missingPackageList;

my $model;
my $great;
my $oslevel;
my $oslevels;
my $h;
my $f;
my $osmodel;
my $hardware;
my $oskey;
my $osver;     # This actually holds the KERNEL version, not the OS version!
my $osVersionMaj; # This holds the OS version's major number, so 15 for SLES 15.2
my $osVersionMin; # This holds the OS version's minor number, so 2 for SLES 15.2
my $kernelVersionMaj; # This holds the OS kernel's major number, so 5 for SLES 15.2 (running 5.3)
my $kernelVersionMin; # This holds the OS kernel's minor number, so 3 for SLES 15.2 (running 5.3)
my $osrev;
my $database;
my $dbfile;

print "\n" if $debug;
print "Welcome to the Domino OS Patch Checker Perl Script...\n" if $debug;
print "This will check the Operating System level and tell you what is missing.\n" if $debug;
print "Note, no patch list if all patches are present.\n" if $debug;
print "\n" if $debug;

&GetOptions();

# See if they only wanted help.
if ($h)
{
	&Help();
}

if (!$database)
{
	$dbfile = "./checkos.dat";
}
else
{
	$dbfile = $database;
}

print "Using dbfile $dbfile\n" if $debug;

&GetOSRev(\$osmodel,\$hardware,\$oskey,\$osver,\$osrev);

$osmodel  =~ s/^\s*//;   # remove leading white
$osmodel  =~ s/\s*$//;   # remove trailing white
$hardware =~ s/^\s*//;   # remove leading white
$hardware =~ s/\s*$//;   # remove trailing white

my $osDetail = $osmodel;
my $os_hit = 0; # no matches for that os yet
my $patchesmissing = 0;

if($osmodel eq "Linux" )
{
	$osDetail = GetDetailLinuxOSInformation($hardware);
}

print "osmodel: $osmodel\n" if $debug;
print "osver: $osver\n" if $debug;
print "osrev: $osrev\n" if $debug;
print "oskey: $oskey\n" if $debug;
print "osDetail: $osDetail\n" if $debug;

print "\nCalling loadCurentPackages\n" if $debug;
&loadCurrentPackages();
print "Back from loadCurentPackages\n" if $debug;

# Load in the messages and packages in the dat file.
print "\nCalling iterateDatFile\n" if $debug;
&iterateDatFile();
print "Back from iterateDatFile\n\n" if $debug;

# If we're on AIX, check the version of the operating system.
if ($osmodel eq "AIX")
{
	print "\nCalling checkAIXVersion\n" if $debug;
	&checkAIXVersion();
	print "Back from checkAIXVersion\n\n" if $debug;
}

# Check that we have the needed packages.
foreach my $package (@requiredPackageList)
{
	print "\nCalling checkPackageIsInstalled\n" if $debug;
	&checkPackageIsInstalled($package); 
	print "Back from checkPackageIsInstalled\n\n" if $debug;
}

print "\nDone looping through packages\n\n" if $debug;

if ($os_hit == 0)
{
	#
	# No matches for this OS 
	#
	print "Warning:\n";
	print "\t\t[$osmodel]  [$oskey]\n";
	print "is not supported\n\n";	 

	exit $patchesmissing;	   
}

my $homedir = $ENV{'HOME'};
print "\n      OS: $osmodel $oskey\n" if $debug;

#
# If there were any missing packages, list them.
#
if ($patchesmissing > 0)
{
	print "\nThe following OS Patches/Packages are missing:\n";
	
	foreach my $package (@missingPackageList)
	{
		# We don't really have a way to seperate mandatory patches from 'good to have'.  Do that here.
		my $freetype = $package =~ /^freetype/s;

		if ($freetype)
		{
			print "Package " . $package . " is not installed.  File viewers in Verse might not work properly.\n";
		}
		else
		{
			print "$package\n";
		}
	}
	print "\n";
}
else
{
	#
	# Only print if debug is enabled - we don't want to see this in silent mode.
	#
	print "\nThe OS appears to have the correct patches\n\n" if $debug;
}	

print "Exited with patchesmissing: $patchesmissing\n" if $debug;

exit $patchesmissing;

#
# End of main program
#

#
# Subroutines are:
#
# sub loadCurrentPackages
# sub iterateDatFile
# sub checkPackageIsInstalled
# sub checkAIXVersion
# sub GetOSRev
# sub GetOptions
# sub GetDetailLinuxOSInformation
# sub Help
#


########################
#
# loadCurrentPackages
#
# Load the current packages we have into 'current' list.
#
########################
sub loadCurrentPackages
{
	my $command = "";
	my $fullcommand = "";

	if ($osmodel eq "AIX")
	{
		$command = "lslpp -Lcq";
	}
	elsif ($osmodel eq "Linux")
	{
	    if (-e "/bin/rpm")
	    {
		$command = "/bin/rpm -q -a";
	    }
	    else
	    {
		print "WARNING: /bin/rpm not found!\n";
		print "WARNING: \n";
		print "WARNING: Ensure you have the equivalent of these packages:\n";
		print "WARNING: \n";
		print "WARNING: glibc-2.17-222.7.el7.x86_64\n";
		print "WARNING: libstdc++-4.8.5-28.el7.x86_64\n";
		
		exit 0;
	    }
	}

	$fullcommand = $command . " 2>&1 |";
	print "loadCurrentPackages: fullcommand = $fullcommand\n" if $debug;

	open CURRENTPATCHES, $fullcommand or die "can't fork: $!";

	while (<CURRENTPATCHES>)
	{
		my $patchno;
		my @current;

		if ($osmodel eq "AIX")
		{
			@current = split(/:/, $_);

			# Concatenate the package with the version with a dash - will match Linux style then
			$patchno = $current[1] . "-" . $current[2];
		} 
		elsif ($osmodel eq "Linux")
		{
			@current = split(/\t+ +| +\t+| +|\t+/, $_);
			$patchno = $current[0];

			#
			# Only push 64-bit packages...
			#
			if (defined($patchno))
			{
				if (index($patchno, "x86_64") == -1)
				{
					next;
				}
			}
		}
		
		if (defined($patchno))
		{
			chomp($patchno);

			# Place this current package into the 'current' list
			push (@currentPackageList, $patchno);
		}
	}

	print "Current 64-bit package count: ", scalar @currentPackageList, "\n\n" if $debug;

	close CURRENTPATCHES or die "bad $command:$! $?";
}

########################
#
# iterateDatFile
#
# Query current patch levels and process against requirements database.
#
########################
sub iterateDatFile
{
	print "Iterating through dat file\n" if $debug;

	if (open(DBFILE, $dbfile))
	{
		my $line = <DBFILE>;
		while (defined($line))
		{
			# Skip blank lines
			if ($line eq "\n")
			{
				$line = <DBFILE>;
				next;
			}

			# Skip comments which start with a hash
			my $isacomment = $line =~ /^#/s;

			if ($isacomment != 1)
			{
				#
				# Only bother pushing patches that are applicable to this OS
				#
				if (index($line, $osDetail) == 0)
				{
					print "Pushing required package: $line" if $debug;

					# This is a package, put it in the list to be checked.
					push (@requiredPackageList, $line);
				}
			}

			$line = <DBFILE>;
		}
	}
	else
	{
		die "Can NOT open: $dbfile\n";
	}
}

########################
#
# subroutine checkPackageIsInstalled
#
# Make sure we have the passed package name installed
#
########################
sub checkPackageIsInstalled
{
    my ($line) = @_;
    my $answer;
    my $patchOS;
    my $patchVer;
    my $patchVersion;
    my @stuff;

    # Remove newline from end of string.
    chomp $line;
    print "checkPackageIsInstalled: line = $line\n" if $debug;

    #
    # Currently, $line will have a patch string in this format:
    #
    # AIX 7.1.0.0 bos.iocp.rte-7.1.0.19
    #
    # or
    #
    # RedHat 3.10 glibc-2.17-222.7.el7.x86_64
    #
    my $patchno = "";
    
    @stuff = split(/ /, $line);

    $patchOS   = $stuff[0];
    $patchVer  = $stuff[1];
    $candidate = $stuff[2];

    if (defined($candidate))
    {
		# Remove any trailing newline
		chomp $candidate;
    }

	#
	# Skip over APARs in the package list, we check those in checkAIXVersion
	#
	if ($patchOS eq 'AIX')
	{
		# Look for a word boundary that then begins with IY, IV, or IX.
		my $apar = $candidate =~ /\bI[YVX]/;

		print "apar = $apar\n" if $debug;

		if ($apar eq "1")
		{  
			print "Skipping APAR\n" if $debug;
			return;
		}
	}
	
    #
    # Is the patch for our OS?
    #
    if ($patchOS eq "$osDetail")
    {
		print "  This patch is for our OS\n" if $debug;
	
		print "  patchOS = $patchOS\n" if $debug;
		print "  patchVer = $patchVer\n" if $debug;
		print "  candidate = $candidate\n" if $debug;
	
		# We have match, so mark this OS as being hit.
		$os_hit = 1;
	
		#
		# Make sure we're at the same OS major version and at least that minor revision
		#
		my $closeEnough = 0;
	
		if ($patchVer ne "ALL")
		{
			# This will be something like "3.10" or "4.18" on Linux or "7.1" on AIX.
			my @splitString = split /\./, $patchVer;
			my $patchKernelMaj = $splitString[0];
			my $patchKernelMin = $splitString[1];
	
			print "  patchKernelMaj = $patchKernelMaj\n" if $debug;
			print "  patchKernelMin = $patchKernelMin\n" if $debug;
	
			if ($kernelVersionMaj == $patchKernelMaj)
			{
				print "  this patch matches our kernel version major\n" if $debug;
			
				if ($kernelVersionMin >= $patchKernelMin)
				{
					print "  closeEnough = 1\n" if $debug;
					$closeEnough = 1;
				}
				else
				{
					print "  this patch does NOT match our kernel version minor\n" if $debug;
					return;
				}
			}
			else
			{
				print "  this patch does NOT match our kernel version major\n" if $debug;
				return;
			}
		}
	    
		if (($patchVer eq "ALL") || $closeEnough)
		{
			#
			# All systems go, let's check that the package is installed!
			#
			my @pieces;
			my $patchpresent = 0;
			my $AIX_OK;
			my $patchTool = "";

			#
			# Split the package name into pieces deliminated by dashes.
			# Note we're not done.  A package name can (and do!) contain a dash.
			#
			@pieces = split(/[-]+/, $candidate);

			print "    Found " . scalar @pieces . " pieces\n" if $debug;

			foreach my $piece (@pieces)
			{
				print "    Current piece = $piece\n" if $debug;
		
				# If it's numeric, we're past the name and have found the version.
				if ($piece =~ /^[0-9]/)
				{
					$patchVersion = $piece;
					print "    patchVersion = $patchVersion\n" if $debug;
					last;
				}
				else
				{
					if ($patchTool eq "")
					{
						$patchTool = $piece;
						print "    patchTool = $patchTool\n" if $debug;
					}
					else
					{
						$patchTool = $patchTool . "-" . $piece;
						print "    patchTool = $patchTool\n" if $debug;
					}
				}
			}
	    
			print "    Final patchTool = $patchTool\n" if $debug;
			print "    Final patchVersion = $patchVersion\n" if $debug;

			# Let's iterate over our currentPackageList to see if this package is installed
			foreach my $package (@currentPackageList)
			{
				#print "Looking at $package\n" if $debug;
		
				# 
				# See if we can rule this out quickly.  Does this package name start with the correct string?
				#
				if ($package =~ /^$patchTool/)
				{
					my $patchToolFound = "";
					my $patchVersionFound = "";
					print "Found potential match in $package\n" if $debug;
		    
					@pieces = split(/[-]+/, $package);

					foreach my $piece (@pieces)
					{
						print "Looking at piece [$piece]\n" if $debug;
							
						# If it's numeric, we're past the name and have found the version.
						if ($piece =~ /^[0-9]/)
						{
							$patchVersionFound = $piece;
							last;
						}
						else
						{
							if ($patchToolFound eq "")
							{
								$patchToolFound = $piece;
							}
							else
							{
								$patchToolFound = $patchToolFound . "-" . $piece;
							}
						}
					}
		    
					print "patchToolFound = $patchToolFound\n" if $debug;
					print "patchVersionFound = $patchVersionFound\n" if $debug;
		    
					if ($patchTool eq $patchToolFound)
					{
						if ($patchVersion eq $patchVersionFound)
						{
							print "Versions match, we're good!\n" if $debug;
							$patchpresent = 1;
			    
							# Exit foreach
							last;
						}
			
						# Compare as strings as we want "1.6.3" to be newer than "1.5.321"
						if ($patchVersion lt $patchVersionFound)
						{
							print "Newer version found, we're good!\n" if $debug;
							$patchpresent = 1;
			    
							# Exit foreach
							last;
						}
						else
						{
							print "Older version found, an update will be needed\n" if $debug;
							last;
						}
					}
					else
					{
						print "Wrong patch tool\n" if $debug;
					}
				}
				else
				{
					# Not even close to matching...
					next;
				}
			} # foreach $package
	    
			if ($patchpresent eq 0)
			{
				if ($osmodel eq 'AIX')
				{
					if (($oslevels >=  $great) || ($AIX_OK == 1) )
					{
						if ($patchTool =~ /IV29604/)
						{
							print "Skipping $patchTool for $oslevel on AIX\n" if $debug;
						}
						else
						{
							$patchesmissing++;
							$line =~ s/\"/ /;
								
							print "ERROR: Missing patch: $candidate \n" if $debug;
			    
							# Place this missing package into the 'missing' list
							push (@missingPackageList, $candidate);
						}
					}
				}
				else
				{
					$patchesmissing++;
					$line =~ s/\"/ /;
		    
					print "ERROR: Missing patch: $candidate \n" if $debug;
		    
					# Place this missing package into the 'missing' list
					push (@missingPackageList, $candidate);
				}
			}
		}
    }
    else
    {
		print "We are not running $patchOS, skipping check\n" if $debug;
    }
}

########################
#
# checkAIXVersion
#
#  Check for correct minimum version of OS
#  Check APARs needed
#
########################
sub checkAIXVersion
{
	select (STDOUT);

	#
	# CHANGE_EACH_RELEASE (if supported version of AIX has changed)
	#
	# As of 12.01 we support AIX 7.2 TL1+.   We may change this in the future.
	#
	# Get current AIX OS TL (technology level)
	#
	my $minTL = "720001";
	my $dispMinTL = "7200-01";

	#
	# First we use "oslevel -r" to check AIX version and TL.  From the man page for oslevel:
	#
	# oslevel -r  Applies all flags to technology levels.
	# 
	# oslevel -r will return something like "7200-04".  The tr command removes the hyphen leaving us with "720004"
	#
	$oslevel =`/bin/oslevel -r| /bin/tr -d '-' 2> /dev/null`;

	chomp $oslevel;
	print "oslevel = $oslevel\n" if $debug;

	if ($oslevel < $minTL)
	{  
		print "The minimum  Technology Level is " . $dispMinTL . ". To check yours, run as root /usr/bin/oslevel -r\n";
		exit $patchesmissing;
	}

	#
	# CHANGE_EACH_RELEASE (if supported version of AIX has changed)
	#
	# We've already checked OS and TL level with "oslevel -r".
	# Now we check it again using "oslevel -s".    From the man page for oslevel:
	#
	# oslevel -s  Applies all flags to service packs. The service pack level returned is in the format 6100-00-01-0748,
	#             where 6100 refers to base level 6.1.0.0; 00 refers to technology level 0; 01 refers to service pack 1;
	#             and 0748 is the build sequence identifier, which is used to determine valid technology levels and service
	#             packs that can be applied to the current level. Attempts to apply a technology level or service pack with a
	#             lower build sequence identifier will fail.  If no flags are specified, the base system software is entirely
	#             at or above the level that is listed in the output of the oslevel command.
	#
	# We currently have no need for checking service packs, but this check may be useful in the future.
	#
	$great = "720001011642";
	my $dispGreat = "7200-01-01-1642";

	#
	# oslevel -s will return something like "7200-01-01-0748"
	#
	# Where 7200 refers to base level 7.2.0.0; 01 refers to technology level 1; 01 refers to service pack 1;
	# and 0748 is the build sequence identifier
	#
	# The tr command removes the hyphen leaving us with "720001010748"
	#
	$oslevels =`/bin/oslevel -s| /bin/tr -d '-' 2> /dev/null`;

	chomp $oslevels;
	print "oslevels = $oslevels\n" if $debug;

	if ($oslevels < $great)
	{
		print "The minimum  Service Pack level is " . $dispGreat . ". To check yours, run as root /usr/bin/oslevel -s\n";
		exit $patchesmissing;	
	}

	print "With AIX we check for APARS and patches separately, you could be missing one and not the other.\n" if $debug;

	if (open (DBFILE, $dbfile))
	{
		my $line = <DBFILE>;
		while (defined($line))
		{
			# Comment lines start with a pound sign.  Skip them.
			if ($line =~ /^#/)
			{
				$line = <DBFILE>;
				next;
			}
			
			# Skip blank lines too.
			if ($line eq "\n")
			{
				$line = <DBFILE>;
				next;
			}

			# Skip if not AIX
			if (! ($line =~ /^AIX/))
			{
				$line = <DBFILE>;
				next;
			}

			chomp $line;
			print "line = $line\n" if $debug;

			my $fields1;
			my @fields1 = split (/ /, $line);

			# Look for a word boundary that then begins with IY, IV, or IX.
			my $apar = $line =~ /\bI[YVX]/;

			print "apar = $apar\n" if $debug;

			if ($apar eq "1")
			{  
				# Yes, this is an APAR
				my $oslevel = `oslevel`;
				
				chop $oslevel;
				print "oslevel = $oslevel\n" if $debug;
				
				if ($fields1[1] eq "$oslevel")
				{ 
					# This APAR matches our version
					my $first_time = 1;
					my $ape = "/usr/sbin/instfix -i -k $fields1[2] 2>&1 |";

					open CURRENTAPARS, $ape or die "can't fork: $!";
					while (<CURRENTAPARS>)
					{
						#
						# If the APAR is not installed this will contain something like:
						#
						#     "There was no data for IJ15678 in the fix database."
						#
						if ($_ =~ / no /)
						{
							if ($first_time == 1 )
							{
								print "You are missing the following AIX APARS:\n";
								$first_time = 0;
							}
							print "$fields1[3]\n";
						}
					} 
					close CURRENTAPARS;
				}
			}

			# Grab the next line and loop
			$line = <DBFILE>;
		}
	}
	else 
	{
		die "Can NOT open: $dbfile\n";
	}
}

########################
#
# subroutine GetOSRev
#
# Determine OS configuration.
#
#  Syntax: &GetOSRev(\$osmodel,\$hardware,\$oskey,\$osver,\$osrev);
#
#	return: 1 on success
#	        0 if unable to determine any attribute
#
#	$osmodel, $hardware, are are strings 
#	$oskey is the vendor's OS version string
#	$osver & $osrev are numeric representations of $oskey
#
#
########################
sub GetOSRev
{
	my ($rModel, $rMachine, $rRevString, $rRevMajor, $rRevMinor) = @_;

	# in:	  -
	# out:	  $rModel,
	#		  $rMachine,
	#		  $rRevString,
	#		  $rRevMajor,
	#		  $rRevMinor
	# return: 1 on success
	#		  0 if unable to determine any attribute
	#
	# $rModel     == AIX | Linux
	# $rMachine   == INTEL | 700 | 800 | Power
	# $rRevString == output of "uname -r" (oslevel on aix)
	# $rRevMajor  == major os rev number
	# $rRevMinor  == canonical 6 digit rep of os minor rev
	# example:	revString	major		minor
	#			5.6			5			060000
	#			4.3.1.0		4			030100
	#			B.11.0		11			000000
	$$rModel		=
	$$rMachine		=
	$$rRevString	=
	$$rRevMajor		=
	$$rRevMinor		=
		"unknown";

	my $revString;
	my $revMinor;

	$model = `uname -s`;

	#
	# If uname failed, bail.
	#
	if ($?)
	{
		die "Calling uname returned $?\n";
	}

	chomp($model);

	if ($model eq "AIX")
	{
		$$rModel = "AIX";
		$$rMachine = "Power";
		$revString = `oslevel`;

		($?) && exit 0;

		chomp($revString);
		$$rRevString = $revString;

		# Break apart into an array using '.' as a deliminator.
		my @splitString = split /\./, $revString;

		print "splitString is @splitString\n" if $debug;

		$kernelVersionMaj = $splitString[0];
		$kernelVersionMin = $splitString[1];

		print "kernelVersionMaj is $kernelVersionMaj\n" if $debug;
		print "kernelVersionMin is $kernelVersionMin\n" if $debug;
	}
	elsif ($model eq "Linux")
	{
		$$rModel = $model;
		$$rMachine = `uname -m`;
		$revString = `uname -r`;

		#
		# If uname failed, bail.
		#
		if ($?)
		{
		    die "Calling uname returned $?\n";
		}

		chomp($revString);
		$$rRevString = $revString;
	}

	print "revString = $revString\n" if $debug;

	local($_) = $revString;

	# From the beginning of the line, match any non digit characters and remove them.
	s/^[^\d]*//;

	# From the beginning of the line, match any digits and extract.  This is the major version.
	($$rRevMajor) = /^(\d*)/;
	print "Rev Major is $$rRevMajor \n" if $debug;

	# From the beginning of the line, remove any digits.
	s/^(\d*)//;
	
	# Match the start of the line being a period (escaped), then extract the match of the following digits.
	if (/^\.(\d*)/)
	{
		$revMinor = sprintf("%02d", /^\.(\d*)/);
		print "Rev Minor is $revMinor\n" if $debug;
	}

	# From the beginning of the line, match a period and any digits and remove.
	s/^\.(\d*)//;

	# Match the start of the line being a period (escaped), then extract the match of the following digits.
	# This is appended to the current minor revision.... which seems odd.
	if (/^\.(\d*)/)
	{
		$revMinor = sprintf("%s%02d", $revMinor, /^\.(\d*)/);
		print "Rev Minor is now $revMinor\n" if $debug;
	}
	
	# From the beginning of the line, match a period and any digits and remove.
	s/^\.(\d*)//;

	# Again, match the start of the line being a period (escaped), then extract the match of the following digits.
	# This is appended to the current minor revision.... which still seems odd.
	if (/^\.(\d*)/)
	{
		$revMinor = sprintf("%s%02d", $revMinor, /^\.(\d*)/);
		print "Rev Minor is now $revMinor\n" if $debug;
	}

	$$rRevMinor = $revMinor;

	return 1;
}

########################
#
# subroutine GetOptions
#
# Parse the command line options.
#
#  Syntax: &GetOptions();
#
#
########################
sub GetOptions
{
	my $numArgs = $#ARGV + 1;
	my $iii = 0;
	$h = $f = 0;

	print "\nYou said: \n" if $debug;
	print @ARGV if $debug;
	print "\n" if $debug;
	print "Number of args is $numArgs\n" if $debug;

	for ($iii=0 ; $iii < $numArgs ; $iii++)
	{
		if ($ARGV[$iii] eq "-h") 
		{
			print "GetOptions: Help\n" if $debug;
			$h = 1;
		}
		elsif ($ARGV[$iii] eq "-d")
		{
			print "GetOptions: Debug\n";
			$debug = 1;
		}
		elsif ($ARGV[$iii] eq "-f") 
		{ 
			$f = 1;
			$database = $ARGV[$iii+1];
			print "GetOptions: File $database\n" if $debug;
		}
	}
	return;
}

sub GetDetailLinuxOSInformation
{
    #
    # Default is 'Linux', if we're a tested platform, we are more specific
    #
	my $os_detail = "Linux";

	#
	# Grab OS name and version from os-release
	#
	my $osName = `grep PRETTY_NAME /etc/os-release`;
	chop $osName;
	print "osName = $osName\n" if $debug;

	#
	# The $os_detail variable is used to match distros in checkos.dat
	#
	if ($osName =~ /Red Hat/)
	{
	    $os_detail = "RedHat";        
		print "Detected RedHat\n" if $debug;
	}
	elsif ($osName =~ /SUSE/)
	{
	    $os_detail = "SuSE";
		print "Detected SuSE\n" if $debug;
	}

	#
	# Find the OS version, like openSUSE 15.2
	#
	# The line in os-release will be like:
	#
	# VERSION_ID="15.2"
	#
	my $osV = `grep VERSION_ID /etc/os-release`;
	chop $osV;
	print "osV= $osV\n" if $debug;

	my $osVersion;
	my $firstQuote = 0;

	$firstQuote = index $osV, "\"";

	if ($firstQuote > 0)
	{
	    # Grab what's inside the quotes
	    $osVersion = substr $osV, $firstQuote + 1;
	    chop $osVersion;
	    print "osVersion = $osVersion\n" if $debug;
	}
	else
	{
	    $osVersion = 0;
	    print "osVersion = $osVersion\n" if $debug;
	}

	my $dot = index $osVersion, ".";

	if ($dot != -1)
	{
	    # Split on '.', so 15.2 will be 15 and 2
	    my @fields = split /\./, $osVersion;
	    $osVersionMaj = $fields[0];
	    $osVersionMin = $fields[1];
	    
	    print "osVersionMaj = $osVersionMaj\n" if $debug;
	    print "osVersionMin = $osVersionMin\n" if $debug;
	}	    
	else
	{
	    $osVersionMaj = 0;
	    $osVersionMin = 0;

	    print "osVersionMaj = $osVersionMaj\n" if $debug;
	    print "osVersionMin = $osVersionMin\n" if $debug;
	}

	#
	# Find the kernel's major and minor versions.
	# The command "uname -r" will return something like:
	#
	# 3.10.0-1127.19.1.el7.x86_64
	#
	# Kernel version:  3
	# Major revision: 10
	#
	# This is as much as we care about.
	#
	my $unameR = `uname -r`;
	chop $unameR;
	print "unameR is $unameR\n" if $debug;

	# Break apart into an array using '.' as a deliminator.
	my @splitString = split /\./, $unameR;

	print "splitString is @splitString\n" if $debug;

	$kernelVersionMaj = $splitString[0];
	$kernelVersionMin = $splitString[1];

	print "kernelVersionMaj is $kernelVersionMaj\n" if $debug;
	print "kernelVersionMin is $kernelVersionMin\n" if $debug;

	return $os_detail;
}

sub Help
{
	#
	# Start help 
	#
	print "\nOption '-h' detected, here are the command line options for checkos.pl\n";
	print "\n";
	print "-f <name>:      Use <name> as patch database\n";
	print "-d:             Print debugging information\n";
	print "-h:             Show the command line options\n";
	print "\n";
	
	exit 0;
}
